//-----------------------------------------------------------------------------
// VCM Plugins - VCM_EnemyInfo
//-----------------------------------------------------------------------------


/*:
@plugindesc Allows the player to use a command that opens a window showing information about all enemies in battle. Window and amount of information are both customizable.
@author VCM Plugins


@param 1 - Enemy Info Command

@param Enemy Info Command Vocabulary
@parent 1 - Enemy Info Command
@desc Text of the Enemy Info Command
Default: Enemy Info
@default Enemy Info

@param Party Command
@parent 1 - Enemy Info Command
@type boolean
@on Add
@off Ignore
@desc Enemy Info Command will be added to the Party Command?
Default: true
@default true

@param Actor Command
@parent 1 - Enemy Info Command
@type boolean
@on Add
@off Ignore
@desc Enemy Info Command will be added to the Actor Command?
Default: false
@default false


@param 2 - Enemy Info Settings

@param Enemy Info x Position
@parent 2 - Enemy Info Settings
@type number
@desc x Position of the Enemy Info Window.
Default: 0
@default 0

@param Enemy Info y Position
@parent 2 - Enemy Info Settings
@type number
@desc y Position of the Enemy Info Window.
Default: 0
@default 0

@param Enemy Info Width
@parent 2 - Enemy Info Settings
@type number
@desc Width of the Enemy Info Window.
Default: 816
@default 816

@param Enemy Info Height
@parent 2 - Enemy Info Settings
@type number
@desc Height of the Enemy Info Window.
Default: 624
@default 624

@param Enemy Info Windowskin
@parent 2 - Enemy Info Settings
@type file
@dir img/system/
@require 1
@desc Windowskin of the Enemy Info Window.
Default: Window
@default Window

@param Enemy Info Opacity
@parent 2 - Enemy Info Settings
@type number
@desc Opacity of the Enemy Info Window.
Default: 255
@default 255

@param Enemy Info Rows
@parent 2 - Enemy Info Settings
@type number
@desc Max Rows of the Enemy Info Window in selection area.
Default: 2
@default 2

@param Enemy Info Columns
@parent 2 - Enemy Info Settings
@type number
@desc Max Columns of the Enemy Info Window in selection area.
Default: 4
@default 4

@param Enemy Info Separators
@parent 2 - Enemy Info Settings
@type boolean
@on Add
@off Ignore
@desc Separators will be added to the Enemy Info Window?
Default: true
@default true

@param Enemy Info Separator Color
@parent 2 - Enemy Info Settings
@type number
@desc Color of the Separator of the Enemy Info Window.
Default: 16
@default 16

@param Enemy Info Terms Color
@parent 2 - Enemy Info Settings
@type number
@desc Color of Labels and Terms of the Enemy Info Window.
Default: 16
@default 16


@param 3 - Enemy Info Description

@param Show Description
@parent 3 - Enemy Info Description
@type boolean
@on Yes
@off No
@desc Description will be shown in Enemy Info Window?
Default: true
@default true

@param Enemy Description
@parent 3 - Enemy Info Description
@type note[]
@desc Description text of each enemy.
Default: ["\"Weak\"","\"Clever\"","\"Nasty\"","\"Bad\""]
@default ["\"Weak\"","\"Clever\"","\"Nasty\"","\"Bad\""]

@param Description Label Vocabulary
@parent 3 - Enemy Info Description
@desc Text of the Description Label in Enemy Info Window.
Default: Description
@default Description

@param Description x Position
@parent 3 - Enemy Info Description
@type number
@desc x Position of the Description.
Default: 10
@default 10

@param Description y Position
@parent 3 - Enemy Info Description
@type number
@desc y Position of the Description.
Default: 438
@default 438

@param Description Width
@parent 3 - Enemy Info Description
@type number
@desc Width of the Description.
Default: 440
@default 440

@param Description Height
@parent 3 - Enemy Info Description
@type number
@desc Height of the Description.
Default: 144
@default 144


@param 4 - Enemy Info Parameters

@param Show Parameters
@parent 4 - Enemy Info Parameters
@type boolean
@on Yes
@off No
@desc Parameters will be shown in Enemy Info Window?
Default: true
@default true

@param Show Parameters Gauges
@parent 4 - Enemy Info Parameters
@type boolean
@on Yes
@off No
@desc Gauges will be shown whenever possible?
Default: true
@default true

@param Parameters Label Vocabulary
@parent 4 - Enemy Info Parameters
@desc Text of the Parameters Label in Enemy Info Window.
Default: Parameters
@default Parameters

@param Parameters Font Size
@parent 4 - Enemy Info Parameters
@type number
@desc Font Size of the Parameters.
Default: 28
@default 28

@param Parameters x Position
@parent 4 - Enemy Info Parameters
@type number
@desc x Position of the Parameters.
Default: 10
@default 10

@param Parameters y Position
@parent 4 - Enemy Info Parameters
@type number
@desc y Position of the Parameters.
Default: 100
@default 100

@param Parameters Width
@parent 4 - Enemy Info Parameters
@type number
@desc Width of the Parameters.
Default: 200
@default 200

@param Parameters Line Height
@parent 4 - Enemy Info Parameters
@type number
@desc Line Height of the Parameters.
Default: 32
@default 32

@param Parameters Height
@parent 4 - Enemy Info Parameters
@type number
@desc Height of the Parameters.
Default: 320
@default 320


@param 5 - Enemy Info Element Rates

@param Show Element Rates
@parent 5 - Enemy Info Element Rates
@type boolean
@on Yes
@off No
@desc Element Rates will be shown in Enemy Info Window?
Default: true
@default true

@param Show Element Rates Icons
@parent 5 - Enemy Info Element Rates
@type boolean
@on Yes
@off No
@desc Icon of Element Rates will be shown?
Default: false
@default false

@param Element Rates Label Vocabulary
@parent 5 - Enemy Info Element Rates
@desc Text of the Element Rates Label in Enemy Info Window.
Default: Element Rates
@default Element Rates

@param Element Rates Font Size
@parent 5 - Enemy Info Element Rates
@type number
@desc Font Size of the Element Rates.
Default: 28
@default 28

@param Element Rates x Position
@parent 5 - Enemy Info Element Rates
@type number
@desc x Position of the Element Rates.
Default: 240
@default 240

@param Element Rates y Position
@parent 5 - Enemy Info Element Rates
@type number
@desc y Position of the Element Rates.
Default: 100
@default 100

@param Element Rates Width
@parent 5 - Enemy Info Element Rates
@type number
@desc Width of the Element Rates.
Default: 200
@default 200

@param Element Rates Line Height
@parent 5 - Enemy Info Element Rates
@type number
@desc Line Height of the Element Rates.
Default: 32
@default 32

@param Element Rates Height
@parent 5 - Enemy Info Element Rates
@type number
@desc Height of the Element Rates.
Default: 320
@default 320


@param 6 - Enemy Info Skills

@param Show Skills
@parent 6 - Enemy Info Skills
@type boolean
@on Yes
@off No
@desc Skills will be shown in Enemy Info Window?
Default: true
@default true

@param Show Skills Icons
@parent 6 - Enemy Info Skills
@type boolean
@on Yes
@off No
@desc Icon of Skills will be shown?
Default: true
@default true

@param Skills Label Vocabulary
@parent 6 - Enemy Info Skills
@desc Text of the Skills Label in Enemy Info Window.
Default: Skills
@default Skills

@param Skills Font Size
@parent 6 - Enemy Info Skills
@type number
@desc Font Size of the Skills.
Default: 28
@default 28

@param Skills x Position
@parent 6 - Enemy Info Skills
@type number
@desc x Position of the Skills.
Default: 480
@default 480

@param Skills y Position
@parent 6 - Enemy Info Skills
@type number
@desc y Position of the Skills.
Default: 100
@default 100

@param Skills Width
@parent 6 - Enemy Info Skills
@type number
@desc Width of the Skills.
Default: 290
@default 290

@param Skills Line Height
@parent 6 - Enemy Info Skills
@type number
@desc Line Height of the Skills.
Default: 32
@default 32

@param Skills Height
@parent 6 - Enemy Info Skills
@type number
@desc Height of the Skills.
Default: 192
@default 192


@param 7 - Enemy Info Actions

@param Show Actions
@parent 7 - Enemy Info Actions
@type boolean
@on Yes
@off No
@desc Actions will be shown in Enemy Info Window?
Default: true
@default true

@param Show Actions Icons
@parent 7 - Enemy Info Actions
@type boolean
@on Yes
@off No
@desc Icon of the Skill of the Actions will be shown?
Default: true
@default true

@param Actions Label Vocabulary
@parent 7 - Enemy Info Actions
@desc Text of the Actions Label in Enemy Info Window.
Default: Actions
@default Actions

@param Actions Font Size
@parent 7 - Enemy Info Actions
@type number
@desc Font Size of the Actions.
Default: 28
@default 28

@param Actions x Position
@parent 7 - Enemy Info Actions
@type number
@desc x Position of the Actions.
Default: 480
@default 480

@param Actions y Position
@parent 7 - Enemy Info Actions
@type number
@desc y Position of the Actions.
Default: 324
@default 324

@param Actions Width
@parent 7 - Enemy Info Actions
@type number
@desc Width of the Actions.
Default: 290
@default 290

@param Actions Line Height
@parent 7 - Enemy Info Actions
@type number
@desc Line Height of the Actions.
Default: 32
@default 32

@param Actions Height
@parent 7 - Enemy Info Actions
@type number
@desc Height of the Actions.
Default: 96
@default 96


@param 8 - Enemy Info States

@param Show States
@parent 8 - Enemy Info States
@type boolean
@on Yes
@off No
@desc States will be shown in Enemy Info Window?
Default: true
@default true

@param Show States Icons
@parent 8 - Enemy Info States
@type boolean
@on Yes
@off No
@desc Icon of the States will be shown?
Default: true
@default true

@param States Label Vocabulary
@parent 8 - Enemy Info States
@desc Text of the States Label in Enemy Info Window.
Default: States
@default States

@param States Font Size
@parent 8 - Enemy Info States
@type number
@desc Font Size of the States.
Default: 28
@default 28

@param States x Position
@parent 8 - Enemy Info States
@type number
@desc x Position of the States.
Default: 480
@default 480

@param States y Position
@parent 8 - Enemy Info States
@type number
@desc y Position of the States.
Default: 442
@default 442

@param States Width
@parent 8 - Enemy Info States
@type number
@desc Width of the States.
Default: 290
@default 290

@param States Line Height
@parent 8 - Enemy Info States
@type number
@desc Line Height of the States.
Default: 32
@default 32

@param States Height
@parent 8 - Enemy Info States
@type number
@desc Height of the States.
Default: 128
@default 128


@param 9 - Enemy Info Element Affinities

@param Show Element Affinities
@parent 9 - Enemy Info Element Affinities
@type boolean
@on Yes
@off No
@desc Element Affinities will be shown in Enemy Info Window?
Default: false
@default false

@param Show Element Affinities Icons
@parent 9 - Enemy Info Element Affinities
@type boolean
@on Yes
@off No
@desc Icon of Element Affinities will be shown?
Default: false
@default false

@param Element Affinities Label Vocabulary
@parent 9 - Enemy Info Element Affinities
@desc Text of the Element Affinities Label in Enemy Info Window.
Default: Element Affinities
@default Element Affinities

@param Element Affinities Font Size
@parent 9 - Enemy Info Element Affinities
@type number
@desc Font Size of the Element Affinities.
Default: 28
@default 28

@param Element Affinities x Position
@parent 9 - Enemy Info Element Affinities
@type number
@desc x Position of the Element Affinities.
Default: 480
@default 480

@param Element Affinities y Position
@parent 9 - Enemy Info Element Affinities
@type number
@desc y Position of the Element Affinities.
Default: 100
@default 100

@param Element Affinities Width
@parent 9 - Enemy Info Element Affinities
@type number
@desc Width of the Element Affinities.
Default: 200
@default 200

@param Element Affinities Line Height
@parent 9 - Enemy Info Element Affinities
@type number
@desc Line Height of the Element Affinities.
Default: 32
@default 32

@param Element Affinities Height
@parent 9 - Enemy Info Element Affinities
@type number
@desc Height of the Element Affinities.
Default: 320
@default 320


@help
-----------------------------------------------------------------------------
Introduction
-----------------------------------------------------------------------------

Version -> 1.03

This plugin was tested only on RPG Maker MV Version 1.6.2.
I cannot guarantee it works on lower versions.

Terms of Use:
 - Available for commercial and non-commercial use
 - You may freely edit the code
 - You are not allowed to redistribute this plugin. Instead,
 provide a link(https://vcm-plugins.itch.io/vcm-enemyinfo)
 - Do not claim this plugin as your own
 - Credit is not required. However, if you want to, credit me as 'VCM Plugins'

This plugin allows the player to use a command that opens a window showing
information about all enemies in battle. Command may be included in Party
Command Window and/or Actor Command Window. Window and amount of information
are both customizable. Information that may be shown include:
Description (text set by the user)
Parameters (HP, MP, TP, ATK, DEF, MAT, MDF, AGI, LUK. AP is supported if
using VCM_ActionPoints)
Element Rates
Skills
Actions (to be used in the current turn)
States
Element Affinities (requires VCM_ElementAffinity)

This documentation contains the following subheaders:
Introduction
Parameters Explanation
Compatibility
Versions


-----------------------------------------------------------------------------
Parameters Explanation
-----------------------------------------------------------------------------

Enemy Info Command Vocabulary
This parameter specifies the text shown to the player in the Enemy Info
Command.

Party Command
This parameter specifies whether the Enemy Info Command will be added to the
Party Command Window or not.

Actor Command
This parameter specifies whether the Enemy Info Command will be added to the
Actor Command Window or not.


Enemy Info x Position
This parameter determines the x position of the Enemy Info Window.
The higher the value, the farther it will be to the right. Non-number values
may cause errors.

Enemy Info y Position
This parameter determines the y position of the Enemy Info Window.
The higher the value, the lower it will be positioned. Non-number values
may cause errors.

Enemy Info Width
This parameter determines the width of the Enemy Info Window. Values lower
than 1 will show nothing. Non-number values may cause errors.

Enemy Info Height
This parameter determines the height of the Enemy Info Window. Values lower
than 1 will show nothing. Non-number values may cause errors.

Enemy Info Windowskin
This parameter specifies which Windowskin you will use for the Enemy Info
Window. The image should be a png file located in the img\system folder.
If setting the value textually, don't use extension. For example,
Window or "Window", works fine, while Window.png or "Window.png", don't.
Invalid values may cause errors.

Enemy Info Opacity
This parameter specifies the opacity of the Enemy Info Window.
Value should be a number between 0 and 255. Any value lower than 0 will be
converted to 0, and any value higher than 255 will become 255.
Higher values means more opaque. With an opacity of 0, the window will become
transparent, but text and icons will still be visible. Non-number values may
cause errors.

Enemy Info Rows
This parameter specifies the maximum number of rows that Enemy Info Window
may have in a single column in the selection area. If this parameter's value
is lower than 1 or non-number, selection area will not be shown, though the
information of the first enemy of the troop will.

Enemy Info Columns
This parameter specifies the maximum number of columns that Enemy Info Window
may have in a single row in the selection area. If this parameter's value is
lower than 1 or non-number, no enemies will be shown, though the information
of the first enemy of the troop will.

Enemy Info Separators
This parameter determines whether separators will be added to the Enemy Info
Window or not. Separators visually delimit the selection area and
information types.

Enemy Info Separator Color
This parameter specifies the color of the separators of the Enemy Info Window.
System colors are used. Non-number values or numbers outside of the 0-31 range
may cause errors.

Enemy Info Terms Color
This parameter specifies the color of the terms and labels of the Enemy Info
Window. System colors are used. Non-number values or numbers outside of the
0-31 range may cause errors.


Show Description
This parameter specifies whether the Description area will be shown in the
Enemy Info Window or not.

Enemy Description
This parameter specifies the text to be shown in the Description area of
an enemy in the Enemy Info Window. Text width is automatically adjusted
to fit and line breaks are valid. This must be set for each enemy,
in the same order of Enemies Database.

Description Label Vocabulary
This parameter specifies the text of the label of the Description area.
It will be placed above the Description area.

Description x Position
This parameter determines the x position of the Description area,
relative to the Enemy Info Window. The higher the value, the farther it will
be to the right. Non-number values may cause errors.

Description y Position
This parameter determines the y position of the Description area,
relative to the Enemy Info Window. The higher the value, the lower it will
be positioned. Non-number values may cause errors.

Description Width
This parameter determines the width of the Description area. Values lower than
1 will show nothing. Non-number values may cause errors.

Description Height
This parameter determines the height of the Description area. Values lower
than 1 will show nothing. Non-number values may cause errors.


Show Parameters
This parameter specifies whether the Parameters area will be shown in the
Enemy Info Window or not.

Show Parameters Gauges
This parameter specifies whether gauges will appear instead of text in the
Parameters area when the values are HP, MP or TP. If using VCM_ActionPoints,
AP is supported too.

Parameters Label Vocabulary
This parameter specifies the text of the label of the Parameters area.
It will be placed above the Parameters area.

Parameters Font Size
This parameter determines the font size of the Parameters area. Values
lower than 1 will show nothing. Non-number values may cause errors.

Parameters x Position
This parameter determines the x position of the Parameters area,
relative to the Enemy Info Window. The higher the value, the farther it will
be to the right. Non-number values may cause errors.

Parameters y Position
This parameter determines the y position of the Parameters area,
relative to the Enemy Info Window. The higher the value, the lower it will
be positioned. Non-number values may cause errors.

Parameters Width
This parameter determines the width of the Parameters area. Values lower than
1 will show nothing. Non-number values may cause errors.

Parameters Line Height
This parameter determines the height of a line in the Parameters area.
If the number of lines multiplied by this value is higher than this plugin's
'Parameters Height' value, not all information will be shown. Non-number
values may cause errors.

Parameters Height
This parameter determines the height of the Parameters area. Values lower
than 1 will show nothing. Non-number values may cause errors.


Show Element Rates
This parameter specifies whether the Element Rates area will be shown in the
Enemy Info Window or not.

Show Element Rates Icons
This parameter specifies whether icons will appear before the element rates
name in the Element Rates area or not. If set to Yes(true), the text will
no longer have its width adjusted to fit and may go out of bounds. Also, 
its color will revert to default.

Element Rates Label Vocabulary
This parameter specifies the text of the label of the Element Rates area.
It will be placed above the Element Rates area.

Element Rates Font Size
This parameter determines the font size of the Element Rates area. Values
lower than 1 will show nothing. Non-number values may cause errors.

Element Rates x Position
This parameter determines the x position of the Element Rates area,
relative to the Enemy Info Window. The higher the value, the farther it
will be to the right. Non-number values may cause errors.

Element Rates y Position
This parameter determines the y position of the Element Rates area,
relative to the Enemy Info Window. The higher the value, the lower it
will be positioned. Non-number values may cause errors.

Element Rates Width
This parameter determines the width of the Element Rates area. Values lower
than 1 will show nothing. Non-number values may cause errors.

Element Rates Line Height
This parameter determines the height of a line in the Element Rates area.
If the number of lines multiplied by this value is higher than this plugin's
'Element Rates Height' value, not all information will be shown. Non-number
values may cause errors.

Element Rates Height
This parameter determines the height of the Element Rates area. Values lower
than 1 will show nothing. Non-number values may cause errors.


Show Skills
This parameter specifies whether the Skills area will be shown in the Enemy
Info Window or not.

Show Skills Icons
This parameter specifies whether icons will appear before the skill name
in the Skills area or not. If set to Yes(true), the text will no longer
have its width adjusted to fit and may go out of bounds.

Skills Label Vocabulary
This parameter specifies the text of the label of the Skills area.
It will be placed above the Skills area.

Skills Font Size
This parameter determines the font size of the Skills area. Values
lower than 1 will show nothing. Non-number values may cause errors.

Skills x Position
This parameter determines the x position of the Skills area,
relative to the Enemy Info Window. The higher the value, the farther it will
be to the right. Non-number values may cause errors.

Skills y Position
This parameter determines the y position of the Skills area,
relative to the Enemy Info Window. The higher the value, the lower it will
be positioned. Non-number values may cause errors.

Skills Width
This parameter determines the width of the Skills area. Values lower than 1
will show nothing. Non-number values may cause errors.

Skills Line Height
This parameter determines the height of a line in the Skills area.
If the number of lines multiplied by this value is higher than this plugin's
'Skills Height' value, not all information will be shown. Non-number
values may cause errors.

Skills Height
This parameter determines the height of the Skills area. Values lower
than 1 will show nothing. Non-number values may cause errors.


Show Actions
This parameter specifies whether the Actions area will be shown in the Enemy
Info Window or not.

Show Actions Icons
This parameter specifies whether icons will appear before the action name
in the Actions area or not. If set to Yes(true), the text will no longer
have its width adjusted to fit and may go out of bounds.

Actions Label Vocabulary
This parameter specifies the text of the label of the Actions area.
It will be placed above the Actions area.

Actions Font Size
This parameter determines the font size of the Actions area. Values
lower than 1 will show nothing. Non-number values may cause errors.

Actions x Position
This parameter determines the x position of the Actions area,
relative to the Enemy Info Window. The higher the value, the farther it will
be to the right. Non-number values may cause errors.

Actions y Position
This parameter determines the y position of the Actions area,
relative to the Enemy Info Window. The higher the value, the lower it will
be positioned. Non-number values may cause errors.

Actions Width
This parameter determines the width of the Actions area. Values lower than 1
will show nothing. Non-number values may cause errors.

Actions Line Height
This parameter determines the height of a line in the Actions area.
If the number of lines multiplied by this value is higher than this plugin's
'Actions Height' value, not all information will be shown. Non-number
values may cause errors.

Actions Height
This parameter determines the height of the Actions area. Values lower
than 1 will show nothing. Non-number values may cause errors.


Show States
This parameter specifies whether the States area will be shown in the Enemy
Info Window or not.

Show States Icons
This parameter specifies whether icons will appear before the state name
in the States area or not. If set to Yes(true), the text will no longer
have its width adjusted to fit and may go out of bounds.

States Label Vocabulary
This parameter specifies the text of the label of the States area.
It will be placed above the States area.

States Font Size
This parameter determines the font size of the States area. Values
lower than 1 will show nothing. Non-number values may cause errors.

States x Position
This parameter determines the x position of the States area,
relative to the Enemy Info Window. The higher the value, the farther it will
be to the right. Non-number values may cause errors.

States y Position
This parameter determines the y position of the States area,
relative to the Enemy Info Window. The higher the value, the lower it will
be positioned. Non-number values may cause errors.

States Width
This parameter determines the width of the States area. Values lower than 1
will show nothing. Non-number values may cause errors.

States Line Height
This parameter determines the height of a line in the States area.
If the number of lines multiplied by this value is higher than this plugin's
'States Height' value, not all information will be shown. Non-number
values may cause errors.

States Height
This parameter determines the height of the States area. Values lower
than 1 will show nothing. Non-number values may cause errors.


Show Element Affinities
This parameter specifies whether the Element Affinities area will be shown
in the Enemy Info Window or not.

Show Element Affinities Icons
This parameter specifies whether icons will appear before the element
affinities name in the Element Affinities area or not. If set to Yes(true),
the text will no longer have its width adjusted to fit and may go out of
bounds. Also,  its color will revert to default.

Element Affinities Label Vocabulary
This parameter specifies the text of the label of the Element Affinities area.
It will be placed above the Element Affinities area.

Element Affinities Font Size
This parameter determines the font size of the Element Affinities area.
Values lower than 1 will show nothing. Non-number values may cause errors.

Element Affinities x Position
This parameter determines the x position of the Element Affinities area,
relative to the Enemy Info Window. The higher the value, the farther it will
be to the right. Non-number values may cause errors.

Element Affinities y Position
This parameter determines the y position of the Element Affinities area,
relative to the Enemy Info Window. The higher the value, the lower it will
be positioned. Non-number values may cause errors.

Element Affinities Width
This parameter determines the width of the Element Affinities area. Values
lower than 1 will show nothing. Non-number values may cause errors.

Element Affinities Line Height
This parameter determines the height of a line in the Element Affinities area.
If the number of lines multiplied by this value is higher than this plugin's
'Element Affinities Height' value, not all information will be shown.
Non-number values may cause errors.

Element Affinities Height
This parameter determines the height of the Element Affinities area. Values
lower than 1 will show nothing. Non-number values may cause errors.


-----------------------------------------------------------------------------
Compatibility
-----------------------------------------------------------------------------

Plugin Manager line-up for maximum compatibility:
VCM_HelpWindow
VCM_PreviousTurn
VCM_BattleSave
VCM_Quicksave
VCM_MovementSpeeds
VCM_StateDescription
VCM_TermDescription
VCM_BattleMainMenu
VCM_EventHighlighting
VCM_SkillBar
VCM_MirroredSpriteset
VCM_ActionPoints
VCM_ActionOrder
VCM_EnemyGauges
VCM_MultipleGauges
VCM_ElementAffinity
VCM_AutoBattle
VCM_EnemyInfo
VCM_NumberBattlers
VCM_NoBattleLog

This plugin uses the current code of the following functions:
Window_PartyCommand.prototype.activate(from Window_Selectable)
Window_PartyCommand.prototype.makeCommandList
Window_ActorCommand.prototype.activate(from Window_Selectable)
Window_ActorCommand.prototype.makeCommandList
Scene_Battle.prototype.update(if using VCM_HelpWindow or VCM_SkillBar)
Scene_Battle_changeInputWindow
Scene_Battle.prototype.createAllWindows
Scene_Battle.prototype.createPartyCommandWindow
Scene_Battle.prototype.createActorCommandWindow

This means that this plugin will use any changes to those functions
made by plugins above it in the Plugin Manager, which may, or may not
be compatible.


-----------------------------------------------------------------------------
Versions
-----------------------------------------------------------------------------

Version -> 1.00
Released Plugin.

Version -> 1.01
Updated documentation.

Version -> 1.02
Fixed a mistake when calculating the limit of enemies that can be selected.
Added some support for use in skills. Updated documentation.

Version -> 1.03
Enemy Info Window will no longer show TP if the 'Display TP in Battle' System
Option is unchecked. Added support for icons for Element Rates and Element
Affinities. Updated documentation.
*/

"use strict";

var Imported = Imported || {};
Imported.VCM_EnemyInfo = true;

function VCMConvert (parameters) {
	function VCMParse (string) {
		try {
			return JSON.parse(string, (key, value) => {
				try {
					return VCMParse(value);
				} catch (e) {
					return value;
				}
			});
		} catch (e) {
			return string;
		}
	};
	return VCMParse(JSON.stringify(parameters));
};

var VCM = VCM || {};
VCM.EnemyInfo = VCM.EnemyInfo || {};
VCM.EnemyInfo = VCMConvert(PluginManager.parameters('VCM_EnemyInfo')) || {};
VCM.EnemyInfoAlias = VCM.EnemyInfoAlias || {};


//-----------------------------------------------------------------------------
// Window_EnemyInfo
//

///
function Window_EnemyInfo() {
    this.initialize.apply(this, arguments);
}

Window_EnemyInfo.prototype = Object.create(Window_Selectable.prototype);
Window_EnemyInfo.prototype.constructor = Window_EnemyInfo;

Window_EnemyInfo.prototype.initialize = function() {
	this._enemies = [];
    Window_Selectable.prototype.initialize.call(this, VCM.EnemyInfo['Enemy Info x Position'], VCM.EnemyInfo['Enemy Info y Position'], VCM.EnemyInfo['Enemy Info Width'], VCM.EnemyInfo['Enemy Info Height']);
	this.opacity = VCM.EnemyInfo['Enemy Info Opacity'];
	this.refresh();
};

Window_EnemyInfo.prototype.loadWindowskin = function() {
    this.windowskin = ImageManager.loadSystem(VCM.EnemyInfo['Enemy Info Windowskin']);
};

Window_EnemyInfo.prototype.numVisibleRows = function() {
    return VCM.EnemyInfo['Enemy Info Rows'];
};

Window_EnemyInfo.prototype.maxCols = function() {
    return VCM.EnemyInfo['Enemy Info Columns'];
};

Window_EnemyInfo.prototype.maxItems = function() {
    return Math.min(this.numVisibleRows() * this.maxCols(), this._enemies.length);
};

Window_EnemyInfo.prototype.enemy = function() {
    return this._enemies[this.index()];
};

Window_EnemyInfo.prototype.enemyIndex = function() {
    var enemy = this.enemy();
    return enemy ? enemy.index() : -1;
};

Window_EnemyInfo.prototype.drawItem = function(index) {
    this.resetTextColor();
    var name = this._enemies[index].name();
    var rect = this.itemRectForText(index);
    this.drawText(name, rect.x, rect.y, rect.width);
	if(VCM.EnemyInfo['Enemy Info Separators']){
		this.contents.fillRect(0, this.numVisibleRows() * this.lineHeight() + this.standardPadding(), this.contents.width, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
	}
};

Window_EnemyInfo.prototype.show = function() {
    this.refresh();
    this.select(0);
    Window_Selectable.prototype.show.call(this);
};

Window_EnemyInfo.prototype.hide = function() {
    Window_Selectable.prototype.hide.call(this);
};

Window_EnemyInfo.prototype.refresh = function() {
	this._enemies = $gameTroop.aliveMembers();
    Window_Selectable.prototype.refresh.call(this);
};

Window_EnemyInfo.prototype.select = function(index, enemy) {
    Window_Selectable.prototype.select.call(this, index);
	if(this.enemy()){
		this.contents.clear();
		if(enemy){
			this._enemies = [enemy];
			this._index = 0;
		}
		else{
			this.refresh();
		}
		if(VCM.EnemyInfo['Show Description']){
			this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			this.drawText(VCM.EnemyInfo['Description Label Vocabulary'], VCM.EnemyInfo['Description x Position'], VCM.EnemyInfo['Description y Position'], VCM.EnemyInfo['Description Width'], 'center');
			this.resetTextColor();
			var i = 0;
			var j = 1;
			if(VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1]){
				while(i < VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].length){
					if((j + 1) * (this.contents.fontSize + 8) > VCM.EnemyInfo['Description Height']){
						i = VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].length;
					}
					else if(VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].indexOf("\n", i) < 0){
						this.drawText(VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].substring(i, VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].length),
						VCM.EnemyInfo['Description x Position'], VCM.EnemyInfo['Description y Position'] + j * (this.contents.fontSize + 8), VCM.EnemyInfo['Description Width']);
						i = VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].length;
					}
					else{
						this.drawText(VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].substring(i, VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].indexOf("\n", i)),
						VCM.EnemyInfo['Description x Position'], VCM.EnemyInfo['Description y Position'] + j * (this.contents.fontSize + 8), VCM.EnemyInfo['Description Width']);
						i = VCM.EnemyInfo['Enemy Description'][this.enemy().enemyId() - 1].indexOf("\n", i) + 1;
					}
					j++;
				}	
			}
			if(VCM.EnemyInfo['Enemy Info Separators']){		
				this.contents.fillRect(VCM.EnemyInfo['Description x Position'], VCM.EnemyInfo['Description y Position'] + this.contents.fontSize + 8,
				VCM.EnemyInfo['Description Width'], 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Description x Position'] - 2, VCM.EnemyInfo['Description y Position'] + VCM.EnemyInfo['Description Height'],
				VCM.EnemyInfo['Description Width'] + 4, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Description x Position'] - 2, VCM.EnemyInfo['Description y Position'] + this.contents.fontSize + 8,
				2, VCM.EnemyInfo['Description Height'] - this.contents.fontSize - 8, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Description x Position'] + VCM.EnemyInfo['Description Width'], VCM.EnemyInfo['Description y Position'] + this.contents.fontSize + 8,
				2, VCM.EnemyInfo['Description Height'] - this.contents.fontSize - 8, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
			}
			this.resetFontSettings();
		}
		if(VCM.EnemyInfo['Show Parameters']){
			this.contents.fontSize = VCM.EnemyInfo['Parameters Font Size'];
			this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			this.drawText(VCM.EnemyInfo['Parameters Label Vocabulary'], VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'], VCM.EnemyInfo['Parameters Width'], 'center');
			i = 1;
			if(VCM.EnemyInfo['Show Parameters Gauges']){
				if((i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.drawActorHp(this.enemy(), VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'], VCM.EnemyInfo['Parameters Width']);
					i++;
				}
				if((i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.drawActorMp(this.enemy(), VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'], VCM.EnemyInfo['Parameters Width']);
					i++;
				}
				if($dataSystem.optDisplayTp && (i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.drawActorTp(this.enemy(), VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'], VCM.EnemyInfo['Parameters Width']);
					i++;
				}
				if(Imported.VCM_ActionPoints && VCM.ActionPoints['Action Points Resource'] === 'AP' && (i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.drawActorAp(this.enemy(), VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'], VCM.EnemyInfo['Parameters Width']);
					i++;
				}
			}
			else{
				if((i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.drawText(TextManager.hpA, VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height']);
					this.resetTextColor();
					this.drawText(this.enemy().hp + '/' + this.enemy().mhp, VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'],
					VCM.EnemyInfo['Parameters Width'], 'right');
					i++;
				}
				if((i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
					this.drawText(TextManager.mpA, VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height']);
					this.resetTextColor();
					this.drawText(this.enemy()._mp + '/' + this.enemy().mmp, VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'],
					VCM.EnemyInfo['Parameters Width'], 'right');
					i++;
				}
				if($dataSystem.optDisplayTp && (i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
					this.drawText(TextManager.tpA, VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height']);
					this.resetTextColor();
					this.drawText(this.enemy()._tp + '/' + this.enemy().maxTp(), VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'],
					VCM.EnemyInfo['Parameters Width'], 'right');
					i++;
				}
				if(Imported.VCM_ActionPoints && VCM.ActionPoints['Action Points Resource'] === 'AP' && (i + 1) * VCM.EnemyInfo['Parameters Line Height'] < VCM.EnemyInfo['Parameters Height']){
					this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
					this.drawText(VCM.ActionPoints['Action Points Vocabulary'], VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height']);
					this.resetTextColor();
					this.drawText(this.enemy()._ap + '/' + VCM.ActionPoints['Max Action Points'], VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'],
					VCM.EnemyInfo['Parameters Width'], 'right');
					i++;
				}
			}
			for(var j = 2; j < this.enemy()._buffs.length; j++){
				if((i + 1) * VCM.EnemyInfo['Parameters Line Height'] > VCM.EnemyInfo['Parameters Height']){
					break;
				}
				this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
				this.drawText(TextManager.param(j), VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height']);
				this.resetTextColor();
				this.drawText(this.enemy().param(j), VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + i * VCM.EnemyInfo['Parameters Line Height'],
				VCM.EnemyInfo['Parameters Width'], 'right');
				i++;
			}
			if(VCM.EnemyInfo['Enemy Info Separators']){
				this.contents.fillRect(VCM.EnemyInfo['Parameters x Position'], VCM.EnemyInfo['Parameters y Position'] + VCM.EnemyInfo['Parameters Line Height'],
				VCM.EnemyInfo['Parameters Width'], 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Parameters x Position'] - 2, VCM.EnemyInfo['Parameters y Position'] + VCM.EnemyInfo['Parameters Height'],
				VCM.EnemyInfo['Parameters Width'] + 4, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Parameters x Position'] - 2, VCM.EnemyInfo['Parameters y Position'] + VCM.EnemyInfo['Parameters Line Height'],
				2, VCM.EnemyInfo['Parameters Height'] - VCM.EnemyInfo['Parameters Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Parameters x Position'] + VCM.EnemyInfo['Parameters Width'], VCM.EnemyInfo['Parameters y Position'] + VCM.EnemyInfo['Parameters Line Height'],
				2, VCM.EnemyInfo['Parameters Height'] - VCM.EnemyInfo['Parameters Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
			}
			this.resetFontSettings();
		}
		if(VCM.EnemyInfo['Show Element Rates']){
			this.contents.fontSize = VCM.EnemyInfo['Element Rates Font Size'];
			this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			this.drawText(VCM.EnemyInfo['Element Rates Label Vocabulary'], VCM.EnemyInfo['Element Rates x Position'], VCM.EnemyInfo['Element Rates y Position'], VCM.EnemyInfo['Element Rates Width'], 'center');
			for(i = 1; i < $dataSystem.elements.length; i++){
				if((i + 1) * VCM.EnemyInfo['Element Rates Line Height'] > VCM.EnemyInfo['Element Rates Height']){
					break;
				}
				if(VCM.EnemyInfo['Show Element Rates Icons']){
					this.drawTextEx($dataSystem.elements[i], VCM.EnemyInfo['Element Rates x Position'], VCM.EnemyInfo['Element Rates y Position'] + i * VCM.EnemyInfo['Element Rates Line Height']);
				}
				else{
					this.drawText($dataSystem.elements[i], VCM.EnemyInfo['Element Rates x Position'], VCM.EnemyInfo['Element Rates y Position'] + i * VCM.EnemyInfo['Element Rates Line Height']);
				}
				this.resetTextColor();
				this.drawText(this.enemy().elementRate(i) * 100 + '%', VCM.EnemyInfo['Element Rates x Position'], VCM.EnemyInfo['Element Rates y Position'] + i * VCM.EnemyInfo['Element Rates Line Height'],
				VCM.EnemyInfo['Element Rates Width'], 'right');
				this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			}
			if(VCM.EnemyInfo['Enemy Info Separators']){
				this.contents.fillRect(VCM.EnemyInfo['Element Rates x Position'], VCM.EnemyInfo['Element Rates y Position'] + VCM.EnemyInfo['Element Rates Line Height'],
				VCM.EnemyInfo['Element Rates Width'], 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Element Rates x Position'] - 2, VCM.EnemyInfo['Element Rates y Position'] + VCM.EnemyInfo['Element Rates Height'],
				VCM.EnemyInfo['Element Rates Width'] + 4, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Element Rates x Position'] - 2, VCM.EnemyInfo['Element Rates y Position'] + VCM.EnemyInfo['Element Rates Line Height'],
				2, VCM.EnemyInfo['Element Rates Height'] - VCM.EnemyInfo['Element Rates Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Element Rates x Position'] + VCM.EnemyInfo['Element Rates Width'], VCM.EnemyInfo['Element Rates y Position'] + VCM.EnemyInfo['Element Rates Line Height'],
				2, VCM.EnemyInfo['Element Rates Height'] - VCM.EnemyInfo['Element Rates Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
			}
			this.resetFontSettings();
		}
		if(VCM.EnemyInfo['Show Skills']){
			this.contents.fontSize = VCM.EnemyInfo['Skills Font Size'];
			this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			this.drawText(VCM.EnemyInfo['Skills Label Vocabulary'], VCM.EnemyInfo['Skills x Position'], VCM.EnemyInfo['Skills y Position'], VCM.EnemyInfo['Skills Width'], 'center');
			this.resetTextColor();
			for(i = 0; i < $dataEnemies[this.enemy().enemyId()].actions.length; i++){
				if((i + 2) * VCM.EnemyInfo['Skills Line Height'] > VCM.EnemyInfo['Skills Height']){
					break;
				}
				if(VCM.EnemyInfo['Show Skills Icons']){
					this.drawTextEx('\\I[' + $dataSkills[$dataEnemies[this.enemy().enemyId()].actions[i].skillId].iconIndex + ']' + $dataSkills[$dataEnemies[this.enemy().enemyId()].actions[i].skillId].name,
					VCM.EnemyInfo['Skills x Position'], VCM.EnemyInfo['Skills y Position'] + (i + 1) * VCM.EnemyInfo['Skills Line Height']);
				}
				else{
					this.drawText($dataSkills[$dataEnemies[this.enemy().enemyId()].actions[i].skillId].name,
					VCM.EnemyInfo['Skills x Position'], VCM.EnemyInfo['Skills y Position'] + (i + 1) * VCM.EnemyInfo['Skills Line Height'], VCM.EnemyInfo['Skills Width']);
				}
			}
			if(VCM.EnemyInfo['Enemy Info Separators']){
				this.contents.fillRect(VCM.EnemyInfo['Skills x Position'], VCM.EnemyInfo['Skills y Position'] + VCM.EnemyInfo['Skills Line Height'],
				VCM.EnemyInfo['Skills Width'], 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Skills x Position'] - 2, VCM.EnemyInfo['Skills y Position'] + VCM.EnemyInfo['Skills Height'],
				VCM.EnemyInfo['Skills Width'] + 4, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Skills x Position'] - 2, VCM.EnemyInfo['Skills y Position'] + VCM.EnemyInfo['Skills Line Height'],
				2, VCM.EnemyInfo['Skills Height'] - VCM.EnemyInfo['Skills Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Skills x Position'] + VCM.EnemyInfo['Skills Width'], VCM.EnemyInfo['Skills y Position'] + VCM.EnemyInfo['Skills Line Height'],
				2, VCM.EnemyInfo['Skills Height'] - VCM.EnemyInfo['Skills Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
			}
			this.resetFontSettings();
		}
		if(VCM.EnemyInfo['Show Actions']){
			this.contents.fontSize = VCM.EnemyInfo['Actions Font Size'];
			this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			this.drawText(VCM.EnemyInfo['Actions Label Vocabulary'], VCM.EnemyInfo['Actions x Position'], VCM.EnemyInfo['Actions y Position'], VCM.EnemyInfo['Actions Width'], 'center');
			this.resetTextColor();
			for(i = 0; i < this.enemy()._actions.length; i++){
				if((i + 2) * VCM.EnemyInfo['Actions Line Height'] > VCM.EnemyInfo['Actions Height']){
					break;
				}
				if(VCM.EnemyInfo['Show Actions Icons']){
					this.drawTextEx('\\I[' + $dataSkills[this.enemy()._actions[i]._item._itemId].iconIndex + ']' + $dataSkills[this.enemy()._actions[i]._item._itemId].name, VCM.EnemyInfo['Actions x Position'],
					VCM.EnemyInfo['Actions y Position'] + (i + 1) * VCM.EnemyInfo['Actions Line Height']);
				}
				else{
					this.drawText($dataSkills[this.enemy()._actions[i]._item._itemId].name, VCM.EnemyInfo['Actions x Position'],
					VCM.EnemyInfo['Actions y Position'] + (i + 1) * VCM.EnemyInfo['Actions Line Height'], VCM.EnemyInfo['Actions Width']);
				}
			}
			if(VCM.EnemyInfo['Enemy Info Separators']){
				this.contents.fillRect(VCM.EnemyInfo['Actions x Position'], VCM.EnemyInfo['Actions y Position'] + VCM.EnemyInfo['Actions Line Height'],
				VCM.EnemyInfo['Actions Width'], 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Actions x Position'] - 2, VCM.EnemyInfo['Actions y Position'] + VCM.EnemyInfo['Actions Height'],
				VCM.EnemyInfo['Actions Width'] + 4, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Actions x Position'] - 2, VCM.EnemyInfo['Actions y Position'] + VCM.EnemyInfo['Actions Line Height'],
				2, VCM.EnemyInfo['Actions Height'] - VCM.EnemyInfo['Actions Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Actions x Position'] + VCM.EnemyInfo['Actions Width'], VCM.EnemyInfo['Actions y Position'] + VCM.EnemyInfo['Actions Line Height'],
				2, VCM.EnemyInfo['Actions Height'] - VCM.EnemyInfo['Actions Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
			}
			this.resetFontSettings();
		}
		if(VCM.EnemyInfo['Show States']){
			this.contents.fontSize = VCM.EnemyInfo['States Font Size'];
			this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			this.drawText(VCM.EnemyInfo['States Label Vocabulary'], VCM.EnemyInfo['States x Position'], VCM.EnemyInfo['States y Position'], VCM.EnemyInfo['States Width'], 'center');
			this.resetTextColor();
			for(i = 0; i < this.enemy().states().length; i++){
				if((i + 2) * VCM.EnemyInfo['States Line Height'] > VCM.EnemyInfo['States Height']){
					break;
				}
				if(VCM.EnemyInfo['Show States Icons']){
					this.drawTextEx('\\I[' + this.enemy().states()[i].iconIndex + ']' + this.enemy().states()[i].name, VCM.EnemyInfo['States x Position'],
					VCM.EnemyInfo['States y Position'] + (i + 1) * VCM.EnemyInfo['States Line Height']);
				}
				else{
					this.drawText(this.enemy().states()[i].name, VCM.EnemyInfo['States x Position'],
					VCM.EnemyInfo['States y Position'] + (i + 1) * VCM.EnemyInfo['States Line Height'], VCM.EnemyInfo['States Width']);
				}
			}
			if(VCM.EnemyInfo['Enemy Info Separators']){
				this.contents.fillRect(VCM.EnemyInfo['States x Position'], VCM.EnemyInfo['States y Position'] + VCM.EnemyInfo['States Line Height'],
				VCM.EnemyInfo['States Width'], 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['States x Position'] - 2, VCM.EnemyInfo['States y Position'] + VCM.EnemyInfo['States Height'],
				VCM.EnemyInfo['States Width'] + 4, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['States x Position'] - 2, VCM.EnemyInfo['States y Position'] + VCM.EnemyInfo['States Line Height'],
				2, VCM.EnemyInfo['States Height'] - VCM.EnemyInfo['States Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['States x Position'] + VCM.EnemyInfo['States Width'], VCM.EnemyInfo['States y Position'] + VCM.EnemyInfo['States Line Height'],
				2, VCM.EnemyInfo['States Height'] - VCM.EnemyInfo['States Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
			}
			this.resetFontSettings();
		}
		if(Imported.VCM_ElementAffinity && VCM.EnemyInfo['Show Element Affinities']){
			this.contents.fontSize = VCM.EnemyInfo['Element Affinities Font Size'];
			this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			this.drawText(VCM.EnemyInfo['Element Affinities Label Vocabulary'], VCM.EnemyInfo['Element Affinities x Position'], VCM.EnemyInfo['Element Affinities y Position'],
			VCM.EnemyInfo['Element Affinities Width'], 'center');
			for(i = 1; i < $dataSystem.elements.length; i++){
				if((i + 1) * VCM.EnemyInfo['Element Affinities Line Height'] > VCM.EnemyInfo['Element Affinities Height']){
					break;
				}
				if(VCM.EnemyInfo['Show Element Affinities Icons']){
					this.drawTextEx($dataSystem.elements[i], VCM.EnemyInfo['Element Affinities x Position'], VCM.EnemyInfo['Element Affinities y Position'] + i * VCM.EnemyInfo['Element Affinities Line Height']);
				}
				else{
					this.drawText($dataSystem.elements[i], VCM.EnemyInfo['Element Affinities x Position'], VCM.EnemyInfo['Element Affinities y Position'] + i * VCM.EnemyInfo['Element Affinities Line Height']);
				}
				this.resetTextColor();
				this.drawText(this.enemy()._elementAffinity[i] * 100 + '%', VCM.EnemyInfo['Element Affinities x Position'],
				VCM.EnemyInfo['Element Affinities y Position'] + i * VCM.EnemyInfo['Element Affinities Line Height'], VCM.EnemyInfo['Element Affinities Width'], 'right');
				this.changeTextColor(this.textColor(VCM.EnemyInfo['Enemy Info Terms Color']));
			}
			if(VCM.EnemyInfo['Enemy Info Separators']){
				this.contents.fillRect(VCM.EnemyInfo['Element Affinities x Position'], VCM.EnemyInfo['Element Affinities y Position'] + VCM.EnemyInfo['Element Affinities Line Height'],
				VCM.EnemyInfo['Element Affinities Width'], 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Element Affinities x Position'] - 2, VCM.EnemyInfo['Element Affinities y Position'] + VCM.EnemyInfo['Element Affinities Height'],
				VCM.EnemyInfo['Element Affinities Width'] + 4, 2, this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Element Affinities x Position'] - 2, VCM.EnemyInfo['Element Affinities y Position'] + VCM.EnemyInfo['Element Affinities Line Height'],
				2, VCM.EnemyInfo['Element Affinities Height'] - VCM.EnemyInfo['Element Affinities Line Height'], this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
				this.contents.fillRect(VCM.EnemyInfo['Element Affinities x Position'] + VCM.EnemyInfo['Element Affinities Width'], VCM.EnemyInfo['Element Affinities y Position']
				+ VCM.EnemyInfo['Element Affinities Line Height'], 2, VCM.EnemyInfo['Element Affinities Height'] - VCM.EnemyInfo['Element Affinities Line Height'],
				this.textColor(VCM.EnemyInfo['Enemy Info Separator Color']));
			}
			this.resetFontSettings();
		}
	}
};
///

//-----------------------------------------------------------------------------
// Window_PartyCommand
//
// The window for selecting whether to fight or escape on the battle screen.

VCM.EnemyInfoAlias.Window_PartyCommand_activate = Window_PartyCommand.prototype.activate;
Window_PartyCommand.prototype.activate = function() {
	///
	if(!VCM.EnemyInfo['Party Command'] || (SceneManager._scene._enemyInfoWindow && !SceneManager._scene._enemyInfoWindow.visible)){
	///
		VCM.EnemyInfoAlias.Window_PartyCommand_activate.call(this);
	///
	}
	///
};

VCM.EnemyInfoAlias.Window_PartyCommand_makeCommandList = Window_PartyCommand.prototype.makeCommandList;
Window_PartyCommand.prototype.makeCommandList = function() {
    VCM.EnemyInfoAlias.Window_PartyCommand_makeCommandList.call(this);
	///
	if(VCM.EnemyInfo['Party Command']){
		if(Imported.VCM_HelpWindow){
			this.addCommand(VCM.EnemyInfo['Enemy Info Command Vocabulary'], 'info', true, null,
			$gameSystem._VCMHelpWindow['Party Command Help Window Text (Battle)'][$gameSystem._VCMHelpWindow['Party Command Help Window Text (Battle)'].length > 1
			&& this._helpWindowTextIndex < $gameSystem._VCMHelpWindow['Party Command Help Window Text (Battle)'].length ? this._helpWindowTextIndex : 0]);
		}
		else{
			this.addCommand(VCM.EnemyInfo['Enemy Info Command Vocabulary'], 'info', true);
		}
	}
	///
};

//-----------------------------------------------------------------------------
// Window_ActorCommand
//
// The window for selecting an actor's action on the battle screen.

VCM.EnemyInfoAlias.Window_ActorCommand_activate = Window_ActorCommand.prototype.activate;
Window_ActorCommand.prototype.activate = function() {
	///
	if(!VCM.EnemyInfo['Actor Command'] || (SceneManager._scene._enemyInfoWindow && !SceneManager._scene._enemyInfoWindow.visible)){
	///
		VCM.EnemyInfoAlias.Window_ActorCommand_activate.call(this);
	///
	}
	///
};

VCM.EnemyInfoAlias.Window_ActorCommand_makeCommandList = Window_ActorCommand.prototype.makeCommandList;
Window_ActorCommand.prototype.makeCommandList = function() {
	VCM.EnemyInfoAlias.Window_ActorCommand_makeCommandList.call(this);
	///
    if (this._actor) {
		if(VCM.EnemyInfo['Actor Command']){
			if(Imported.VCM_HelpWindow){
				this.addCommand(VCM.EnemyInfo['Enemy Info Command Vocabulary'], 'info', true, null,
				$gameSystem._VCMHelpWindow['Actor Command Help Window Text (Battle)'][$gameSystem._VCMHelpWindow['Actor Command Help Window Text (Battle)'].length > 1 
				&& this._helpWindowTextIndex < $gameSystem._VCMHelpWindow['Actor Command Help Window Text (Battle)'].length ? this._helpWindowTextIndex : 0]);
			}
			else{
				this.addCommand(VCM.EnemyInfo['Enemy Info Command Vocabulary'], 'info', true);
			}
		}
	}
	///
};

//-----------------------------------------------------------------------------
// Scene_Battle
//
// The scene class of the battle screen.

if(Imported.VCM_HelpWindow || Imported.VCM_SkillBar){
	VCM.EnemyInfoAlias.Scene_Battle_update = Scene_Battle.prototype.update;
	Scene_Battle.prototype.update = function() {
		VCM.EnemyInfoAlias.Scene_Battle_update.call(this);
		///
		if(this._enemyInfoWindow && this._enemyInfoWindow.visible){
			if(this._helpWindow){
				this._helpWindow.visible = false;
			}
			if(this._skillBarHelpWindow){
				this._skillBarHelpWindow.visible = false;
			}
			if(this._skillSwapHelpWindow){
				this._skillSwapHelpWindow.visible = false;
			}
			if(this._skillBar){
				this._skillBar.visible = false;
			}
			if(this._skillSwap){
				this._skillSwap.visible = false;
			}
		}
		///
	};
}

VCM.EnemyInfo.Scene_Battle_changeInputWindow = Scene_Battle.prototype.changeInputWindow;
Scene_Battle.prototype.changeInputWindow = function() {
	VCM.EnemyInfo.Scene_Battle_changeInputWindow.call(this);
	///
	if(this._enemyInfoWindow && this._enemyInfoWindow.visible){
		this._partyCommandWindow.deactivate();
		this._actorCommandWindow.deactivate();
		if(this._actionOrderWindow){
			this._actionOrderWindow.visible = false;
		}
	}
	///
};

VCM.EnemyInfoAlias.Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
Scene_Battle.prototype.createAllWindows = function() {
    VCM.EnemyInfoAlias.Scene_Battle_createAllWindows.call(this);
	///
	this._enemyInfoWindow = new Window_EnemyInfo();
	this._enemyInfoWindow.setHandler('ok',     this.onEnemyInfo.bind(this));
    this._enemyInfoWindow.setHandler('cancel', this.onEnemyInfo.bind(this));
	this.addWindow(this._enemyInfoWindow);
	this._enemyInfoWindow.hide();
	///
};

VCM.EnemyInfoAlias.Scene_Battle_createPartyCommandWindow = Scene_Battle.prototype.createPartyCommandWindow;
Scene_Battle.prototype.createPartyCommandWindow = function() {
    VCM.EnemyInfoAlias.Scene_Battle_createPartyCommandWindow.call(this);
	///
	if(VCM.EnemyInfo['Party Command']){
		this._partyCommandWindow.setHandler('info', this.commandEnemyInfo.bind(this));
	}
	///
};

VCM.EnemyInfoAlias.Scene_Battle_createActorCommandWindow = Scene_Battle.prototype.createActorCommandWindow;
Scene_Battle.prototype.createActorCommandWindow = function() {
    VCM.EnemyInfoAlias.Scene_Battle_createActorCommandWindow.call(this);
	///
	if(VCM.EnemyInfo['Actor Command']){
		this._actorCommandWindow.setHandler('info', this.commandEnemyInfo.bind(this));
	}
	///
};

///
Scene_Battle.prototype.commandEnemyInfo = function(enemy) {
	this._enemyInfoWindow.show();
	this._enemyInfoWindow.activate();
	if(enemy){
		this._enemyInfoWindow.select(0, enemy);
	}
};
///

///
Scene_Battle.prototype.onEnemyInfo = function() {
	this._enemyInfoWindow.hide();
};
///